/***************************************************************************
                          LOW_linkDS2480B.cpp  -  description
                             -------------------
    begin                : Sat Jul 13 2002
    copyright            : (C) 2002 by Harald Roelle, Helmut Reiser
    email                : roelle@informatik.uni-muenchen.de, reiser@informatik.uni-muenchen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <memory>


#include "LOW_linkDS2480B.h"
#include "LOW_helper_msglog.h"

#include "LOW_platformMisc.h"
#include "LOW_IPCKeyGeneratorFactory.h"
#include "LOW_portSerialFactory.h"
#include "LOW_semaphoreSetFactory.h"
#include "LOW_sharedMemSegmentFactory.h"


LOW_linkDS2480B::LOW_linkDS2480B( const LOW_portSerialFactory::portSpecifier_t &inSerPortSpec, const RXPOL_val_t inRXPOL, 
                                  const bool inHasExternalPower, const bool inAllowProgPulse) :
  LOW_link( true, true, inHasExternalPower, inAllowProgPulse),
  receivePolarity( inRXPOL)
{
  initCmdAvailTable();

  serialPort   = LOW_portSerialFactory::new_portSerial( inSerPortSpec);
  
  auto_ptr<LOW_IPCKeyGenerator> keyGenerator (LOW_IPCKeyGeneratorFactory::new_IPCKeyGenerator());

  semSet       = LOW_semaphoreSetFactory::new_semaphoreSet( keyGenerator->getSemSetKey( inSerPortSpec), semaphoreCount, 1);
  
  sharedMemSeg = LOW_sharedMemSegmentFactory::new_sharedMemSegment( keyGenerator->getSharedMemKey( inSerPortSpec), sizeof( sharedAttr_t));
  sharedAttr = (sharedAttr_t*)sharedMemSeg->get();
  
  sharedAttr->wireSpeed = flexible_speed;
  resetLinkAdapter();
  
  // configure the recommeded optimal parameters as of app note #148
  setPullDownSlewRate( PDSR_1_37);
  setWrite1LowTime( W1LT_11);
  setSampleOffsetWrite0Rec( SOW0RT_10);
}

  
LOW_linkDS2480B::~LOW_linkDS2480B()
{
  serialPort->tty_flush();
  
  delete sharedMemSeg;
  delete semSet;
  delete serialPort;
}


//=====================================================================================
//
// touch data methods
//
 
bool LOW_linkDS2480B::touchBit( const bool inSendBit, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( inPullup != pullUp_NONE ) {
    setStrongPullupDuration_cmd( strongPullup_2_SPUD_val( inPullup));
  }
  
  return singleBit_cmd( inSendBit, (inPullup==pullUp_NONE)?false:true);
}

 
uint8_t LOW_linkDS2480B::touchByte( const uint8_t inSendByte, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( inPullup != pullUp_NONE ) {
    setStrongPullupDuration_cmd( strongPullup_2_SPUD_val( inPullup));
    pulse_cmd( true, false, true);
  }
  
  setMode( data_mode);
  
  serialPort->tty_write( inSendByte);
  if ( inSendByte == SwitchToCommandMode_Cmd )
    serialPort->tty_write( inSendByte);
  
  uint8_t retValue = serialPort->tty_readByte( (inPullup==pullUp_NONE)?false:true);
      
  if ( inPullup != pullUp_NONE ) {
    pulse_cmd( false, false, true);
  }
  
  return retValue;
}

  
byteVec_t LOW_linkDS2480B::touchBlock( const byteVec_t &inBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  byteVec_t            retValue   = byteVec_t( inBytes.size());
  byteVec_t            writeBytes;
    
  /* // This breaks on RedHat 7.2:
   * byteVec_t            writeBytes = byteVec_t( inBytes);
   * byteVec_t::iterator  iter = writeBytes.begin();
   * while( iter != writeBytes.end() ) {
   *  if ( *iter == SwitchToCommandMode_Cmd )
   *    writeBytes.insert( iter, SwitchToCommandMode_Cmd);
   *   iter++;
   * }
   */
   
  for( unsigned int a=0; a<inBytes.size(); a++) {
    writeBytes.push_back( inBytes[a]);
    if ( inBytes[a] == SwitchToCommandMode_Cmd )
      writeBytes.push_back( inBytes[a]);
  }
  
  if ( inPullup != pullUp_NONE ) {
    setStrongPullupDuration_cmd( strongPullup_2_SPUD_val( inPullup));
    pulse_cmd( true, false, true);
  }
  
  setMode( data_mode);

  serialPort->tty_write( writeBytes);
  serialPort->tty_read( retValue, (inPullup==pullUp_NONE)?false:true);
      
  if ( inPullup != pullUp_NONE ) {
    pulse_cmd( false, false, true);
  }

  return retValue;
}

  
  
//=====================================================================================
//
// read data methods
//
  
bool LOW_linkDS2480B::readDataBit( const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  return touchBit( true, inPullup);
}


uint8_t LOW_linkDS2480B::readDataByte( const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  return touchByte( 0xff, inPullup);
}


void LOW_linkDS2480B::readData( byteVec_t &outBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  byteVec_t  sendBytes = byteVec_t( outBytes.size(), 0xff);
  byteVec_t  recBytes;
  
  recBytes = touchBlock( sendBytes, inPullup);
  for( unsigned int a=0; a<recBytes.size(); a++)
    outBytes[a] = recBytes[a];
}


    
//=====================================================================================
//
// write data methods
//
 
void LOW_linkDS2480B::writeData( const bool inSendBit, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( touchBit( inSendBit, inPullup) != inSendBit )
    throw comm_error( "Response not equal to sent bit", __FILE__, __LINE__);
}


void LOW_linkDS2480B::writeData( const uint8_t inSendByte, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( touchByte( inSendByte, inPullup) != inSendByte )
    throw comm_error( "Response not equal to sent byte", __FILE__, __LINE__);
}


void LOW_linkDS2480B::writeData( const byteVec_t &inSendBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  byteVec_t readVec = touchBlock( inSendBytes, inPullup);
  
  for( unsigned int i=0; i<inSendBytes.size(); i++)
    if ( readVec[i] != inSendBytes[i] ) {
      throw comm_error( "Response not equal to sent byte", __FILE__, __LINE__);
    }
}

  
    
//=====================================================================================
//
// misc methods
//
  
void LOW_linkDS2480B::resetLinkAdapter()
{
  commLock lock( *this);
  
  sharedAttr->internalMode = command_mode;

  //
  // begin with the standard speed
  //
  serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit8_size, 
                             LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, 
                             LOW_portSerial::B9600_speed);
  
  serialPort->tty_flush(); // flush the buffers
    
  serialPort->tty_break(); // send a break to reset the DS2480
  LOW_platformMisc::milliSleep( 2); // delay to let line settle

  serialPort->tty_flush(); // flush the buffers
  
  //
  // let the DS2480 adapt to the serial speed
  //
  serialPort->tty_write( SerialSpeedAdapt_Cmd);
  serialPort->tty_flush( false, true);
  LOW_platformMisc::milliSleep( 4);
  
  //
  // configure desired serial speed and polarity
  //
  RBR_val_t serialSpeed;
  switch ( sharedAttr->wireSpeed ) {
    case normal_speed:     serialSpeed = RBR_9_6;   break;
    case flexible_speed:   serialSpeed = RBR_9_6;   break;
    case overdrive_speed:  serialSpeed = RBR_57_6;  break;
    default:               throw comm_error( "Unknown wire speed detected", __FILE__, __LINE__);
  }
  try {
    setRS232BaudRate_cmd( serialSpeed, receivePolarity);
  }                // ignore errors, result can't be right
  catch ( ...) {}  // because we're just setting the correct baud rate
  
  //
  // now switch the port to the new speed
  //
  switch ( sharedAttr->wireSpeed ) {
    case normal_speed:
      serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit8_size, 
                                 LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, 
                                 LOW_portSerial::B9600_speed);
      break;
    
    case flexible_speed:
      serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit8_size, 
                                 LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, 
                                 LOW_portSerial::B9600_speed);
      break;
    
    case overdrive_speed:
      serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit8_size, 
                                 LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, 
                                 LOW_portSerial::B57600_speed);
      break;
    
    default:
      throw comm_error( "Unknown wire speed detected", __FILE__, __LINE__);
  }
  
  // do it again, to test wheter it was successfull
  setRS232BaudRate_cmd( serialSpeed, receivePolarity);
}  


bool LOW_linkDS2480B::resetBus()
{
  commLock lock( *this);
  
  resetAnswer_t  answer;
  
  reset_cmd( &answer);

  hasProgramPulse = answer.isVppPresent;

  /** @todo Check if and how "alarming presence pulse" response works. */  
  switch( answer.busStatus ) {
    case oneWireShorted_busStat:        throw comm_error( "Short to ground detected", __FILE__, __LINE__);
    case noPresencePulse_busStat:       return false;
    case presencePulse_busStat:         // fall thru
    case alarmingPresencePulse_busStat: // fall thru
    default:                            return true;
  }
}

 
void LOW_linkDS2480B::strongPullup( const unsigned long inMilliSecs)
{
  commLock lock( *this);
  
  setStrongPullupDuration_cmd( SPUD_inf);
  
  pulse_cmd( false, false, false);
  LOW_platformMisc::milliSleep( inMilliSecs); 
  serialPort->tty_write( PulseTermination_Cmd);
}

    
void LOW_linkDS2480B::programPulse()
{
  commLock lock( *this);
  
  if ( ! allowProgPulse )
    throw notAllowed_error( "Program pulse not allowed", __FILE__, __LINE__);

  if ( ! hasProgramPulse )
    throw illegalLevel_error( "Program pulse not available", __FILE__, __LINE__);
   
  setProgPulseDuration_cmd( PPD_512);
  pulse_cmd( false, true, false);
}


void LOW_linkDS2480B::doSearchSequence( const LOW_deviceIDRaw &inBranchVector, 
                                        LOW_deviceIDRaw &outFoundID, LOW_deviceIDRaw &outDiscrVec)
{
  commLock lock( *this);
  
  byteVec_t  sendVector = byteVec_t( 16, 0x00);

  int bitPos = 0;
  for( int a=0; a<16; a++) {
    for( int b=0; b<4; b++) {
      if ( inBranchVector.getBit( bitPos)==true )
        sendVector[a] |= 0x01<<(2*b+1);
      
      bitPos++;
    }
  }
    
  searchAccelCtrl_cmd( true);
  byteVec_t recVector = touchBlock( sendVector);
  searchAccelCtrl_cmd( false);
  
  if ( recVector.size()!=16 )
    throw sizeMismatch_error( "Search received vector size mismatch", __FILE__, __LINE__);
  
  bitPos = 0;
  for( int a=0; a<16; a++) {
    for( int b=0; b<4; b++) {
      if ( recVector[a] & (0x01<<((2*b)+1)) )
        outFoundID.setBit( bitPos, true);
      else
        outFoundID.setBit( bitPos, false);
      
      if ( recVector[a] & (0x01<<(2*b)) )
        outDiscrVec.setBit( bitPos, true);
      else
        outDiscrVec.setBit( bitPos, false);
      
      bitPos++;
    }  
  }
  
}


//=====================================================================================
//
// DS2480B specific methods
//
  
void LOW_linkDS2480B::setWireSpeed( const wireSpeed_t inWireSpeed)
{
  commLock lock( *this);
  
  sharedAttr->wireSpeed = inWireSpeed;
  
  resetLinkAdapter();
  resetBus();
}


LOW_linkDS2480B::wireSpeed_t LOW_linkDS2480B::getWireSpeed()
{
  commLock lock( *this);

  return sharedAttr->wireSpeed;
}


void LOW_linkDS2480B::setPullDownSlewRate( const PDSR_val_t inPDSR)
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only configurable in flexible speed mode", __FILE__, __LINE__);
    
  setPullDownSlewRate_cmd( inPDSR);
}


LOW_linkDS2480B::PDSR_val_t LOW_linkDS2480B::getPullDownSlewRate()
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only allowed in flexible speed mode", __FILE__, __LINE__);
  
  return getPullDownSlewRate_cmd();
}


void LOW_linkDS2480B::setWrite1LowTime( const W1LT_val_t inW1LT)
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only configurable in flexible speed mode", __FILE__, __LINE__);

  setWrite1LowTime_cmd( inW1LT);
}


LOW_linkDS2480B::W1LT_val_t LOW_linkDS2480B::getWrite1LowTime()
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only allowed in flexible speed mode", __FILE__, __LINE__);
  
  return getWrite1LowTime_cmd();
}


void LOW_linkDS2480B::setSampleOffsetWrite0Rec( const SOW0RT_val_t inSOW0RT)
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only configurable in flexible speed mode", __FILE__, __LINE__);

  setSampleOffsetWrite0Rec_cmd( inSOW0RT);
}


LOW_linkDS2480B::SOW0RT_val_t LOW_linkDS2480B::getSampleOffsetWrite0Rec()
{
  commLock lock( *this);
  
  if ( sharedAttr->wireSpeed != flexible_speed )
    throw notAllowed_error( "Only allowed in flexible speed mode", __FILE__, __LINE__);
  
  return getSampleOffsetWrite0Rec_cmd();
}



//=====================================================================================
//
// DS2480B commands
//


void LOW_linkDS2480B::reset_cmd( resetAnswer_t *outAnswer)
{
  commLock lock( *this);
  
  setMode( command_mode);

  serialPort->tty_flush();
  
  uint8_t outByte = Reset_Cmd | (sharedAttr->wireSpeed&0x03)<<2;
  serialPort->tty_write( outByte);
  
  uint8_t reply = serialPort->tty_readByte();
  
  if ( (reply&0xc0) != (outByte&0xc0) )
    throw comm_error( "Unexpected reply from reset command", __FILE__, __LINE__);

  outAnswer->busStatus     = (busStatus_t)(reply&0x03);
  outAnswer->isVppPresent  = (reply>>5)&0x01;
  outAnswer->chipRevision  = (reply>>2)&0x07;
}


bool LOW_linkDS2480B::singleBit_cmd( const bool inBitVal, const bool inStrongPullup)
{
  commLock lock( *this);
  
  setMode( command_mode);

  uint8_t outByte = SingleBit_Cmd | ((uint8_t)(inBitVal))<<4 | (sharedAttr->wireSpeed&0x03)<<2 | ((uint8_t)(inStrongPullup))<<1;
  serialPort->tty_write( outByte);
  
  uint8_t reply = serialPort->tty_readByte();
  
  if ( (reply&0xfc) != (outByte&0xfc) )
    throw comm_error( "Unexpected reply from single bit command", __FILE__, __LINE__);

  bool retVal = reply&0x01;

  if ( inStrongPullup ) {
    reply = serialPort->tty_readByte();
    if ( (reply&0xfc) != 0xec )
      throw comm_error( "Unexpected reply from single bit strong pullup response", __FILE__, __LINE__);
  }
  
  return retVal;
}


void LOW_linkDS2480B::pulse_cmd( const bool inArm, const bool inProgPulse, const bool inImmidiateTerm)
{
  commLock lock( *this);
  
  setMode( command_mode);

  byteVec_t  outBytes;
  
  outBytes.push_back( Pulse_Cmd | ((uint8_t)(inProgPulse))<<4 | (0x03)<<2 | ((uint8_t)(inArm))<<1);
  if ( inImmidiateTerm )
    outBytes.push_back( PulseTermination_Cmd);
  
  serialPort->tty_write( outBytes);
  
  uint8_t reply = serialPort->tty_readByte();
  
  if ( (reply&0xfc) != (outBytes[0]&0xfc) )
    throw comm_error( "Unexpected reply from pulse command", __FILE__, __LINE__);
}


void LOW_linkDS2480B::searchAccelCtrl_cmd( const bool inAccelOn)
{
  commLock lock( *this);
  
  setMode( command_mode);

  uint8_t outByte = SearchAccel_Cmd | ((uint8_t)(inAccelOn))<<4 | (sharedAttr->wireSpeed&0x03)<<2 ;
  serialPort->tty_write( outByte);
}



//=====================================================================================
//
// DS2480B configuration commands
//
  
void LOW_linkDS2480B::setPullDownSlewRate_cmd( const PDSR_val_t inPDSR)
{
  writeConfigValue( PullDownSlewRate_cfgCmd, inPDSR);
}

LOW_linkDS2480B::PDSR_val_t LOW_linkDS2480B::getPullDownSlewRate_cmd()
{
  return (PDSR_val_t)readConfigValue( PullDownSlewRate_cfgCmd);
}


void LOW_linkDS2480B::setProgPulseDuration_cmd( const PPD_val_t inPPD)
{
  writeConfigValue( ProgPulseDuration_cfgCmd, inPPD);
}

LOW_linkDS2480B::PPD_val_t LOW_linkDS2480B::getProgPulseDuration_cmd()
{
  return (PPD_val_t)readConfigValue( ProgPulseDuration_cfgCmd);
}


void LOW_linkDS2480B::setStrongPullupDuration_cmd( const SPUD_val_t inSPUD)
{
  writeConfigValue( StrongPullupDuration_cfgCmd, inSPUD);
}

LOW_linkDS2480B::SPUD_val_t LOW_linkDS2480B::getStrongPullupDuration_cmd()
{
  return (SPUD_val_t)readConfigValue( StrongPullupDuration_cfgCmd);
}


void LOW_linkDS2480B::setWrite1LowTime_cmd( const W1LT_val_t inW1LT)
{
  writeConfigValue( Write1LowTime_cfgCmd, inW1LT);
}

LOW_linkDS2480B::W1LT_val_t LOW_linkDS2480B::getWrite1LowTime_cmd()
{
  return (W1LT_val_t)readConfigValue( Write1LowTime_cfgCmd);
}


void LOW_linkDS2480B::setSampleOffsetWrite0Rec_cmd( const SOW0RT_val_t inSOW0RT)
{
  writeConfigValue( SampleOffsetWrite0Rec_cfgCmd, inSOW0RT);
}

LOW_linkDS2480B::SOW0RT_val_t LOW_linkDS2480B::getSampleOffsetWrite0Rec_cmd()
{
  return (SOW0RT_val_t)readConfigValue( SampleOffsetWrite0Rec_cfgCmd);
}


void LOW_linkDS2480B::setLoadSensorThreshold_cmd( const LST_val_t inLST)
{
  writeConfigValue( LoadSensorThreshold_cfgCmd, inLST);
}

LOW_linkDS2480B::LST_val_t LOW_linkDS2480B::getLoadSensorThreshold_cmd()
{
  return (LST_val_t)readConfigValue( LoadSensorThreshold_cfgCmd);
}


void LOW_linkDS2480B::setRS232BaudRate_cmd( const RBR_val_t inRBR, const RXPOL_val_t inRXPOL)
{
  writeConfigValue( RS232BaudRate_cfgCmd, ((inRXPOL&0x01)<<2) | (inRBR&0x03) );
}

LOW_linkDS2480B::RBR_val_t LOW_linkDS2480B::getRS232BaudRate_cmd()
{
  return (RBR_val_t)(readConfigValue( RS232BaudRate_cfgCmd) & 0x03);
}

LOW_linkDS2480B::RXPOL_val_t LOW_linkDS2480B::getRS232RxPol_cmd()
{
  return (RXPOL_val_t)((readConfigValue( RS232BaudRate_cfgCmd)>>2) & 0x01);
}



//=====================================================================================
//
// internal methods
//

void LOW_linkDS2480B::writeConfigValue( const uint8_t inParamCode, const uint8_t inParamValue)
{
  commLock lock( *this);
  
  uint8_t outByte       = (inParamCode&0x07)<<4 | (inParamValue&0x07)<<1 | 0x01;
  uint8_t expectedReply = outByte & ~0x01;
  
  setMode( command_mode);

  serialPort->tty_write( outByte);
  if ( serialPort->tty_readByte() != expectedReply )
    throw comm_error( "Unexpected reply while setting config value", __FILE__, __LINE__);
}
  
  
uint8_t LOW_linkDS2480B::readConfigValue( const uint8_t inParamCode)
{
  commLock lock( *this);
  
  uint8_t outByte = (inParamCode&0x07)<<1 | 0x01;
  
  setMode( command_mode);
  
  serialPort->tty_write( outByte);
  
  uint8_t reply = serialPort->tty_readByte();
  if ( (reply&0x01)!=0x00 || (reply&0x70)!=(inParamCode&0x07)<<4 )
    throw comm_error( "Unexpected reply while getting config value", __FILE__, __LINE__);

  return (reply>>1)&0x03;
}


void LOW_linkDS2480B::setMode( const internalMode_t inMode)
{
  if ( sharedAttr->internalMode == inMode )
    return;

  commLock lock( *this);

  if      ( sharedAttr->internalMode==command_mode && inMode==data_mode ) {
    serialPort->tty_write( SwitchToDataMode_Cmd);
    sharedAttr->internalMode = inMode;
  }
  else if ( sharedAttr->internalMode==data_mode && inMode==command_mode ) {
    serialPort->tty_write( SwitchToCommandMode_Cmd);
    sharedAttr->internalMode = inMode;
  }
  else {
    throw internal_error( "Illegal mode transition detected", __FILE__, __LINE__);
  }
}




//=====================================================================================
//
// misc methods
//

LOW_linkDS2480B::SPUD_val_t LOW_linkDS2480B::strongPullup_2_SPUD_val( const strongPullup_t inStrongPullup)
{
  switch( inStrongPullup ) {
    case pullUp_16_4:  return SPUD_16_4;
    case pullUp_65_5:  return SPUD_65_5;
    case pullUp_131:   return SPUD_131;
    case pullUp_262:   return SPUD_262;
    case pullUp_524:   return SPUD_524;
    case pullUp_1048:  return SPUD_1048;
    default:           throw internal_error( "Unknown strongPullup value detected", __FILE__, __LINE__);
  }
}


void LOW_linkDS2480B::initCmdAvailTable()
{
  cfgCmdAvailable[PullDownSlewRate_cfgCmd][normal_speed]         = false;
  cfgCmdAvailable[PullDownSlewRate_cfgCmd][flexible_speed]       = true;
  cfgCmdAvailable[PullDownSlewRate_cfgCmd][overdrive_speed]      = false;

  cfgCmdAvailable[ProgPulseDuration_cfgCmd][normal_speed]        = true;
  cfgCmdAvailable[ProgPulseDuration_cfgCmd][flexible_speed]      = true;
  cfgCmdAvailable[ProgPulseDuration_cfgCmd][overdrive_speed]     = true;

  cfgCmdAvailable[StrongPullupDuration_cfgCmd][normal_speed]     = true;
  cfgCmdAvailable[StrongPullupDuration_cfgCmd][flexible_speed]   = true;
  cfgCmdAvailable[StrongPullupDuration_cfgCmd][overdrive_speed]  = true;

  cfgCmdAvailable[Write1LowTime_cfgCmd][normal_speed]            = false;
  cfgCmdAvailable[Write1LowTime_cfgCmd][flexible_speed]          = true;
  cfgCmdAvailable[Write1LowTime_cfgCmd][overdrive_speed]         = false;

  cfgCmdAvailable[SampleOffsetWrite0Rec_cfgCmd][normal_speed]    = false;
  cfgCmdAvailable[SampleOffsetWrite0Rec_cfgCmd][flexible_speed]  = true;
  cfgCmdAvailable[SampleOffsetWrite0Rec_cfgCmd][overdrive_speed] = false;

  cfgCmdAvailable[LoadSensorThreshold_cfgCmd][normal_speed]      = true;
  cfgCmdAvailable[LoadSensorThreshold_cfgCmd][flexible_speed]    = true;
  cfgCmdAvailable[LoadSensorThreshold_cfgCmd][overdrive_speed]   = true;

  cfgCmdAvailable[RS232BaudRate_cfgCmd][normal_speed]            = true;
  cfgCmdAvailable[RS232BaudRate_cfgCmd][flexible_speed]          = true;
  cfgCmdAvailable[RS232BaudRate_cfgCmd][overdrive_speed]         = true;
}

